/*
 materialize Pagination v0.2.2
 http://mirjamsk.github.io/materialize-pagination/

 Sample usage:
    $('#pagination').materializePagination({ 
        lastPage:  10,
        firstPage:  1,
        urlParameter: 'page',
        useUrlParameter: true,
        onClickCallback: function(requestedPage){
            console.log('Requested page is '+ requestedPage)
        }
    });  
*/
;
(function($, window, document, undefined) {

    var MaterializePagination = function(elem, options) {
        this.$elem = $(elem);
        this.options = options;
        this.$container = null;
        this.$prevEllipsis = null;
        this.$nextEllipsis = null;
        this.currentPage = null;
        this.visiblePages = [];
    };

    MaterializePagination.prototype = {
        defaults: {
            lastPage: 1,
            firstPage: 1,
            urlParameter: 'page',
            useUrlParameter: true,
            maxVisiblePages: 3,
            responsive: false,
            showWords: true,
            smallVersion: false,
            onClickCallback: function() {}
        },

        init: function() {
            // Combine defaults with user-specified options
            this.config = $.extend({}, this.defaults, this.options);

            //make sure maxVisiblePages value is allowed
            this.sanitizeMaxVisiblePages(this.config.maxVisiblePages);

            //check for breakpoints
            this.changeValuesAtBreakpoints(this.config.responsive);

            // Get page defined by the urlParameter
            var requestedPage = this.config.useUrlParameter ? this.parseUrl() : this.config.firstPage;
            
            // Create initial pagination and add it to the DOM
            if (this.createPaginationBase(requestedPage))
                this.bindClickEvent();
                this.disablePrevNext();
        },
        sanitizeMaxVisiblePages: function(maxVisiblePages) {
            var allowedNumbers = [3, 5, 7, 9, 11];
            if (allowedNumbers.indexOf(maxVisiblePages) < 0) {
                console.error('Error: ' + this.config.maxVisiblePages + ' is not a valid value for maxVisiblePages. Valid numbers include 3, 5, 7, 9, 11. The value has been changed to the default value of ' + this.defaults.maxVisiblePages + '.');
                this.config.maxVisiblePages = 3;
            }
        },
        // builds <, 1, ..., ..., 50, >
        // appends pagination ul container to div
        // requests current page based on url parameter
        createPaginationBase: function(requestedPage) { 
            if (isNaN(this.config.firstPage) || isNaN(this.config.lastPage)) {
                console.error('Both firstPage and lastPage attributes need to be integer values');
                return false;
            } else if (this.config.firstPage > this.config.lastPage) {
                console.error('Value of firstPage must be less than the value of lastPage');
                return false;
            }
            this.config.firstPage = parseInt(this.config.firstPage);
            this.config.lastPage = parseInt(this.config.lastPage);

            this.currentPage = this.config.firstPage - this.config.maxVisiblePages;

            this.$container = $('<ul>')
                .addClass('pagination-nav-list');

            this.$prevEllipsis = this.util.Ellipsis();
            this.$nextEllipsis = this.util.Ellipsis();

            var $firstPage = this.util.createPage(this.config.firstPage);
            var $prevChevron = this.util.createChevron('prev');
            var $nextChevron = this.util.createChevron('next');

            this.$container
                .append($prevChevron)
                .append($firstPage)
                .append(this.$prevEllipsis.$elem)
                .append(this.$nextEllipsis.$elem)
                .append($nextChevron);

            if (this.config.lastPage > this.config.firstPage) {
                var $lastPage = this.util.createPage(this.config.lastPage);
                $lastPage.insertBefore($nextChevron);
            }

            this.requestPage(requestedPage, true);
            this.renderActivePage();
            this.$elem.append(this.$container);

            if (this.config.showWords != true) {
                $('.pagination-nav-button--prev-next').addClass('words-hidden');
            }

            if (this.config.smallVersion == true) {
                this.$elem.addClass('pagination-nav--small');
            } else {
                this.$elem.removeClass('pagination-nav--small');
            }

            return true;
        },

        // requests current page based on url parameter
        // attached to each button - <, >, #, 
        requestPage: function(requestedPage, initing) {
            switch (requestedPage) {
                case this.currentPage:
                    return;
                case this.currentPage - 1:
                    this.requestPrevPage();
                    break;
                case this.currentPage + 1:
                    this.requestNextPage();
                    break;
                default:
                    this.requestPageByNumber(requestedPage);
            }
            if (!initing)
                this.config.onClickCallback(this.currentPage);
            this.renderActivePage();

            if (this.config.useUrlParameter)
                this.updateUrlParam(this.config.urlParameter, this.currentPage);
        },

        //changes current page to 1 before it
        //updates visiblePages (3 pages shown before/after/between ellipses (not first or last))
        //removes last visible page, adds new page to front of array 
        requestPrevPage: function() {
            this.currentPage -= 1;
            this.visiblePages.pop().remove();

            switch (this.config.maxVisiblePages) {
                case 5:
                    this.visiblePages.unshift(this.insertPrevPaginationComponent(this.currentPage - 2));
                    break;
                case 7:
                    this.visiblePages.unshift(this.insertPrevPaginationComponent(this.currentPage - 3));
                    break;
                case 9:
                    this.visiblePages.unshift(this.insertPrevPaginationComponent(this.currentPage - 4));
                    break;
                case 11:
                    this.visiblePages.unshift(this.insertPrevPaginationComponent(this.currentPage - 5));
                    break;
                default: //3
                    this.visiblePages.unshift(this.insertPrevPaginationComponent(this.currentPage - 1)); 
            }
        },
        //changes current page to 1 after it
        //updates visiblePages (3 pages shown before/after/between ellipses (not first or last))
        //removes first visible page, adds new page to end of array 
        requestNextPage: function() {
            this.currentPage += 1;
            this.visiblePages.shift().remove();

            switch (this.config.maxVisiblePages) {
                case 5:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 2));
                    break;
                case 7:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 3));
                    break;
                case 9:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 4));
                    break;
                case 11:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 5));
                    break;
                default: //3
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 1));
            }
        },
        //removes all visiblePages
        //updates current page
        //creates all new visiblePages - new current page, one before it, one after it
        requestPageByNumber: function(requestedPage) {
            this.purgeVisiblePages();
            this.currentPage = requestedPage;   

            switch (this.config.maxVisiblePages) {
                case 5:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 2));
                    break;
                case 7:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 3));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 2));
                    break;
                case 9:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 4));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 3));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 2));
                    break;
                case 11:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 5));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 4));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 3));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 2));
                    break;
                default: //3
                    break;
            }

            this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage - 1));
            this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage));
            this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 1));

            switch (this.config.maxVisiblePages) {
                case 5:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 2));
                    break;
                case 7:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 2));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 3));
                    break;
                case 9:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 2));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 3));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 4));
                    break;
                case 11:
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 2));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 3));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 4));
                    this.visiblePages.push(this.insertNextPaginationComponent(this.currentPage + 5));
                    break;
                default: //3
                    break;
            }
        },
        //shows/hides relevant ellipsis buttons
        //removes active from old page, adds it to new active page
        //disables prev/next button if on first/last page respectively
        renderActivePage: function() {
            this.renderEllipsis();
            this.$container.find('li.active').removeClass('active').attr('tabindex', 0);
            var currentPageComponent = $(this.$container.find('[data-page="' + this.currentPage + '"]')[0]);
            currentPageComponent.addClass('active').attr('tabindex', '-1');
            this.disablePrevNext();
        },
        //disables prev/next button if on first/last page respectively
        disablePrevNext: function() {
            if ($('.pagination-nav-button.active').attr('data-page') == this.config.firstPage) {
                $('[data-page="prev"]').addClass('pagination-nav-button-disabled').attr({'disabled': true, 'tabindex': '-1'});
                $('[data-page="next"]').removeClass('pagination-nav-button-disabled').attr({'disabled': false, 'tabindex': 0});
            } else if ($('.pagination-nav-button.active').attr('data-page') == this.config.lastPage) {
                $('[data-page="next"]').addClass('pagination-nav-button-disabled').attr({'disabled': true, 'tabindex': '-1'});
                $('[data-page="prev"]').removeClass('pagination-nav-button-disabled').attr({'disabled': false, 'tabindex': 0});
            } else {
                $('[data-page="prev"]').removeClass('pagination-nav-button-disabled').attr({'disabled': false, 'tabindex': 0});
                $('[data-page="next"]').removeClass('pagination-nav-button-disabled').attr({'disabled': false, 'tabindex': 0});
            }
        },
        //shows/hides relevant ellipsis
        renderEllipsis: function() {
            //if only last ellipsis is showing
            if (this.$prevEllipsis.isHidden &&
                this.currentPage >= this.config.firstPage + this.config.maxVisiblePages - ((this.config.maxVisiblePages - 3) / 2)) { 
                this.$prevEllipsis.show();
            }
            else if (!this.$prevEllipsis.isHidden &&
                this.currentPage <= this.config.firstPage + this.config.maxVisiblePages - (Math.floor(this.config.maxVisiblePages/2))) {
                this.$prevEllipsis.hide();
            }
            if (this.$nextEllipsis.isHidden &&
                this.currentPage <= this.config.lastPage - this.config.maxVisiblePages + ((this.config.maxVisiblePages - 3) / 2)) {
                this.$nextEllipsis.show();
            }
            else if (!this.$nextEllipsis.isHidden &&
                this.currentPage >= this.config.lastPage - this.config.maxVisiblePages + (Math.floor(this.config.maxVisiblePages/2))) {
                this.$nextEllipsis.hide();
            }
        },
        //makes all li buttons clickable, turns pageData for li's into a number, requests the page
        //makes all li buttons work with tab+enter
        bindClickEvent: function() {
            var self = this;
            this.$container.on('click', 'li', function() {
                var requestedPage = self.sanitizePageRequest($(this).data('page'));
                self.requestPage(requestedPage);
            });
            this.$container.on('keyup', 'li', function(event) {
                if (event.keyCode === 13) { //enter key
                    event.preventDefault();
                    this.click();
                }
            });
        },
        //if pageNumber is between first and last, it will be created as a button and inserted after first ellipsis 
        insertPrevPaginationComponent: function(pageNumber) {
            if (pageNumber > this.config.firstPage && pageNumber < this.config.lastPage) {
                var $paginationComponent = this.util.createPage(pageNumber);
                return $paginationComponent.insertAfter(this.$prevEllipsis.$elem);
            }
            return $('');
        },
        //if pageNumber is between first and last, it will be created as a button and inserted before last ellipsis
        insertNextPaginationComponent: function(pageNumber) {
            if (pageNumber > this.config.firstPage && pageNumber < this.config.lastPage) {
                var $paginationComponent = this.util.createPage(pageNumber);
                return $paginationComponent.insertBefore(this.$nextEllipsis.$elem);
            }
            return $('');
        },
        //turns pageData into a number
        sanitizePageRequest: function(pageData) {
            var requestedPage = this.config.firstPage;

            if (pageData === 'prev') {
                requestedPage =
                    this.currentPage === this.config.firstPage ?
                    this.currentPage : this.currentPage - 1;
            } else if (pageData === 'next') {
                requestedPage =
                    this.currentPage === this.config.lastPage ?
                    this.currentPage : this.currentPage + 1;
            } else if (!isNaN(pageData) &&
                pageData >= this.config.firstPage &&
                pageData <= this.config.lastPage) {
                requestedPage = parseInt(pageData);
            }
            return requestedPage;
        },
        //removes all visiblePages
        purgeVisiblePages: function() {
            var size = this.visiblePages.length;
            for (var page = 0; page < size; page += 1)
                this.visiblePages.pop().remove();
        },
        //if this.config.useUrlParameter is true
        //get url based on urlParameter "page"
        //format it into a number and return requested page
        parseUrl: function() {
            var requestedPage = this.getUrlParamByName(this.config.urlParameter) || this.config.firstPage;
            return this.sanitizePageRequest(requestedPage);
        },
        //get url based on urlParameter "page"
        getUrlParamByName: function(name) {
            name = name.replace(/[\[\]]/g, "\\$&");
            var url = window.location.href;
            var regex = new RegExp("[?&]" + name + "(=([^&#]*)|&|#|$)");
            var results = regex.exec(url);
            if (!results) return null;
            if (!results[2]) return '';
            return decodeURIComponent(results[2].replace(/\+/g, " "));
        },
        //create or update query string in url
        updateUrlParam: function(key, value) {
            var baseUrl = [location.protocol, '//', location.host, location.pathname].join(''),
                urlQueryString = document.location.search,
                newParam = key + '=' + value,
                params = '?' + newParam;

            if (urlQueryString) { // If the "search" string exists, then build params from it
                var keyRegex = new RegExp('([\?&])' + key + '[^&]*');

                if (urlQueryString.match(keyRegex) !== null) { // If param exists already, update it
                    params = urlQueryString.replace(keyRegex, "$1" + newParam);
                } else { // Otherwise, add it to end of query string
                    params = urlQueryString + '&' + newParam;
                }
            }
            window.history.pushState('', '', params);
        },
        //breakpoints
        changeValuesAtBreakpoints: function(responsiveOptions) {
            if (!responsiveOptions) {
                return false;
            }
            
            for (var i = 0; i < responsiveOptions.length; i++) {
                var breakpoint = responsiveOptions[i].breakpoint;
                var settings = responsiveOptions[i].settings;
                var windowWidth = $(window).width();

                if (windowWidth <= breakpoint) {
                    for (var prop in settings) {
    
                        switch (prop) {
                            case 'maxVisiblePages':
                                this.config.maxVisiblePages = settings[prop];
                                break;
                            case 'showWords':
                                this.config.showWords = settings[prop];
                                break;
                            case 'smallVersion':
                                this.config.smallVersion = settings[prop];
                                break;
                            default:
                                break;
                        }
                    }
                }               
            }
        },
        util: {
            //creates the li based on the pageNumber given from other functions, adds its data-page attribute
            createPage: function(pageData) {
                return $('<li>')
                    .addClass('pagination-nav-button')
                    .html('<a>' + pageData + '</a>')
                    .attr('data-page', pageData)
                    .attr('tabindex', 0);
            },
            //creates the li for < and >
            createChevron: function(type) {
                var rightSvg = '<svg class="bh-svg-arrow-2-forward" width="24px" height="24px" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><title>bh-ic-arrow-2-forward</title><g stroke="none" stroke-width="1" fill="none" fill-rule="evenodd"><path d="M4,12 C4,12.5522847 4.44771525,13 5,13 L16,13 L16,13 L16,16 L20,12 L16,8 L16,11 L5,11 C4.44771525,11 4,11.4477153 4,12 Z"  fill="#56595D" fill-rule="nonzero"></path></g></svg>';
                var leftSvg = '<svg class="bh-svg-arrow-2-back" width="24px" height="24px" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><title>bh-ic-arrow-2-back</title><g stroke="none" stroke-width="1" fill="none" fill-rule="evenodd"><path d="M4,12 C4,12.5522847 4.44771525,13 5,13 L16,13 L16,13 L16,16 L20,12 L16,8 L16,11 L5,11 C4.44771525,11 4,11.4477153 4,12 Z" fill="#56595D" fill-rule="nonzero" transform="translate(12.000000, 12.000000) scale(-1, 1) translate(-12.000000, -12.000000) "></path></g></svg>';
                var svgToUse = type === 'next' ? rightSvg : leftSvg;
                var textToUse = type === 'next' ? 'Next' : 'Prev';
                var buttonLabel = type === 'next' ? '<span class="pagination-nav-words">' + textToUse + '</span>' + svgToUse : svgToUse + '<span class="pagination-nav-words">' + textToUse + '</span>'; 
                var $icon = $('<span>')
                    .addClass('pagination-nav-button-label')
                    .html(buttonLabel);

                return this.createPage(type).text('')
                    .attr('data-page', type)
                    .addClass('pagination-nav-button--prev-next')
                    .attr('tabindex', 0)
                    .append($icon);
            },
            //creates the li for 2 ellipsis buttons
            Ellipsis: function() {
                var $ellipsis = $('<li>');
                $ellipsis.text('...');
                $ellipsis.addClass('pagination-nav-button pagination-nav-ellipsis hide disabled');
                return {
                    $elem: $ellipsis,
                    isHidden: true,
                    show: function() {
                        this.isHidden = false;
                        this.$elem.removeClass('hide');
                    },
                    hide: function() {
                        this.isHidden = true;
                        this.$elem.addClass('hide');
                    }
                };
            }
        }
    };
    //builds the pagination with given user options
    MaterializePagination.defaults = MaterializePagination.prototype.defaults;
    $.fn.materializePagination = function(options) {
        return this.each(function() {
            new MaterializePagination(this, options).init();
        });
    };
})(jQuery, window, document);